C> \ingroup wfn1
C> @{
C>
C> \brief Choose orbitals to swap to generate a particular state
C>
C> Given a particular orbital occupation for the ground state guesses
C> for all other states can be generated by swapping occupied and
C> virtual orbitals. Allowing for at most swapping one pair of alpha
C> and one pair of beta orbitals we can define the following things.
C>
C> If the number of \f$\alpha\f$- (`nea`) and \f$\beta\f$-electrons 
C> (`neb`) is the same then the symmetry of the state can be maintained
C> by applying the same permutation to both spin channels. This is
C> requested by setting `oclosed = .TRUE.`.
C>
C> Furthermore, state 1 is generated by the identity permutation,
C> subsequently the permutations of the \f$\alpha\f$-orbitals are 
C> counted off (incrementing the virtual orbitals fastest).
C> Next permutations over \f$\alpha\f$ and \f$\beta\f$ orbitals are
C> counted off progressing the \f$\alpha\f$-spin channel fastest.
C>
C> Internally we use:
C> * \f$N_o^a\f$ the number of alpha electron occupied orbitals
C> * \f$N_v^a\f$ the number of alpha electron virtual orbitals
C> * \f$N_o^b\f$ the number of beta electron occupied orbitals
C> * \f$N_v^b\f$ the number of beta electron virtual orbitals
C> * \f$N_o^a N_v^a + 1\f$ the number of alpha electron orbital swaps
C>   including the identity
C> * \f$N_o^b N_v^b + 1\f$ the number of beta electron orbital swaps
C>   including the identity
C> * \f$I_o^a \in [0,N_o^a-1]\f$ the offset relative to the highest
C>   occupied alpha electron orbital
C> * \f$I_v^a \in [0,N_v^a-1]\f$ the offset relative to the lowest
C>   virtual alpha electron orbital
C> * \f$I_o^b \in [0,N_o^b-1]\f$ the offset relative to the highest
C>   occupied beta electron orbital
C> * \f$I_v^b \in [0,N_v^b-1]\f$ the offset relative to the lowest
C>   virtual beta electron orbital
C> * \f$I=I_o^b N_v^b N_v^a N_o^a + I_v^b N_v^a N_o^a + I_o^a N_v^a +
C>   I_v^a + 1\f$ is the number of the state
C>
      subroutine wfn1_swap(nea,neb,nbf,ist,ioa,iva,iob,ivb,oclosed)
      implicit none
c
#include "errquit.fh"
c
      integer nea !< [Input] The number of alpha electrons
      integer neb !< [Input] The number of beta electrons
      integer nbf !< [Input] The number of basis functions
      integer ist !< [Input] The state \f$I\f$
      logical oclosed !< [Input] "Closed shell" requested
                      !< - `.TRUE.` apply the same permutation to both
                      !<   spin channels if they have the same number
                      !<   electrons
                      !< - `.FALSE.` apply different permutation to 
                      !<   different spin channels
      integer ioa !< [Output] The alpha occupied orbital offset
      integer iva !< [Output] The alpha virtual orbital offset
      integer iob !< [Output] The beta occupied orbital offset
      integer ivb !< [Output] The beta virtual orbital offset
c
      integer nva !< The number of alpha virtual orbitals
      integer nvb !< The number of beta virtual orbitals
      integer istate !< The local state counter
c
      nva = nbf - nea
      nvb = nbf - neb
c
      istate = 1
      ioa = nea
      iva = nea
      iob = neb
      ivb = neb
      if (istate.eq.ist) return
c
      if (nea.eq.neb.and.oclosed) then
        do ioa = nea, 1, -1
          do iva = nea+1, nbf
            iob = ioa
            ivb = iva
            istate = istate + 1
            if (istate.eq.ist) return
          enddo
        enddo
      else
        do ioa = nea, 1, -1
          do iva = nea+1, nbf
            istate = istate + 1
            if (istate.eq.ist) return
          enddo
        enddo
        do iob = neb, 1, -1
          do ivb = neb+1, nbf
            do ioa = nea, 1, -1
              do iva = nea+1, nbf
                istate = istate + 1
                if (istate.eq.ist) return
              enddo
            enddo
          enddo
        enddo
      endif
      call errquit("wfn1_swap: ist out of range",ist,UERR)
c
      end
C> @}
