# ISOIMAP -- Draws ellipses taken from a isophote table.
#
# A SDAS table generated by task ELLIPSE is rebinned in a 
# number of sem-major axis values. Task DISPLAY is used to load the 
# image display, hidden task MAP creates a set of tables whith the 
# ellipse points, and these tables are fed to task SGRAPH for overploting 
# on the image display. A maximum of 10 levels is allowed. Since no world
# coordinate system information is available to task MAP, the code
# enforces option fill=yes in DISPLAY. 
#
#						Ivo Busko   3/90
# 26-May-2004  Phil Hodge  Remove code to save/restore parameters.

procedure isoimap (image, table)

char	image     = ""    {prompt="input image"}
char	table     = ""    {prompt="input table"}
bool	fulltable = yes   {prompt="use full range of `SMA' from table ?"}
real	minsma    = 1.    {min=0., prompt="minimum semi-major axis"}
real	maxsma    = 1.    {min=1., prompt="maximum semi-major axis"}
int	nlevels   = 3     {min=2, max=99, prompt="number of levels"}
char	color     = "r"   {prompt="overlay color", enum="r|g|b|y|w"}
int	frame     = 1     {min=1, max=4, prompt="image display frame"}

begin
	file	tab, temp, temp2, ima
	char	list, colour
	int	ncu, frm
	real	min, max, stp
	bool	fta

	# Read task parameters.
	min = 0.
	max = 0.
	ima = image
	tab = table
	fta = fulltable
	min = minsma
	max = maxsma
	ncu = nlevels
	if (ncu > 99) {
	    print ("Maximum of 99 levels !\n")
	    bye
	}
	if (!fta) {
	    if (((max - min) <= 0.) || (max <= 0.) || (min < 0.)) {
	        print ("Error in maxsma, minsma.\n")
	        bye
	    }
	}
	colour = color
	frm = frame

	# Check for the presence of pre-requisite packages.
	if (!deftask("tsort")) {
	    print ("Package 'ttools' must be loaded first !\n")
	    bye
	}
	if (!deftask("sgraph")) {
	    print ("Package 'stplot' or 'tbplot' must be loaded first !\n")
	    bye
	}
	if (!deftask("display")) {
	    print ("Packages 'images' and 'tv'  must be loaded first !\n")
	    bye
	}

	# Do it.

	tsort (tab, "SMA")		# First, sort table

	if (fta) {			# Get minsma and maxsma from table
	    tinfo (tab, ttout=no)
	    tabpar (tab, "SMA", 1)
	    min = real (tabpar.value)
	    tabpar (tab, "SMA", tinfo.nrows)
	    max = real (tabpar.value)
	}

	temp2 = mktemp ("tmp$treb")		# Get rid of flipping PA
	tcopy (tab, temp2, verbose-)
	tcalc (temp2, "PA", "if PA<0. then PA+180. else PA", datatype="real")

	temp = mktemp ("tmp$treb")		# Rebin table
	stp = (max - min) / (ncu - 1)
	trebin (temp2, temp, "SMA", min, max, stp, xtable="",
		function="linear", extrapolate=yes,
		value=INDEF, padvalue=INDEF, verbose=no)

	# this fixes a bug related to the way the display fill mode
	# is implemented. It shifts the image by 1/2 pixel thus we
	# have to shift the ellipse coordinates by the same amount
	# (IB 7/31/96)
	tcalc (temp, "X0", "X0 - 0.5")
	tcalc (temp, "Y0", "Y0 - 0.5")

	# Now call the hidden 'map' task.
	map (ima, temp, "tp", 1)	# Create x,y files

	list = "tp.001"
	for (i = 2; i <= ncu; i=i+1) {
	    if (i < 10)
	        list = list // ",tp.00" // i
	    else
	        list = list // ",tp.0" // i
	}

	display (ima, frm, xcenter=0.5, ycenter=0.5, xsize=1.0, ysize=1.0,
		xmag=1.0, ymag=1.0, fill=yes)

	# call sgraph in a loop to avoid the 14 curve overplot limit of sgraph,
	# and the string length limit of list (63 characters).  Do maximum of 8
	# curves at a time, instead of one at a time to save time.  Also 8 is 
	# a good number since sgraph uses 4 different types of lines. 
	# JC Hsu 11/29/95
	# pltpar.erraxis = 0 added on 11/29/95 to fix a bug, JC Hsu.
	for (j = 1; j <= (ncu-1)/8+1; j=j+1) {
	    if ((j-1)*8+1 < 10)
	        list = "tp.00" // ((j-1)*8+1)
	    else
	        list = "tp.0" // ((j-1)*8+1)
	    for (i = (j-1)*8+2; i <= min(ncu,j*8); i=i+1) {
	        if (i < 10)
	            list = list // ",tp.00" // i
	        else
	            list = list // ",tp.0" // i
	    }

	    # Fix a bug (wr=0, wb=1) here, JC Hsu 6/15/94
	    sgraph (list, device="imd" // colour, append=no, 
		pointmode=no, erraxis=0, box=no,
		left=0, right=1, bottom=0, top=1,
		wl=0, wr=1, wb=0, wt=1)
	}

	# Delete temporary files.
	delete (temp  // ".tab", verify = no)
	delete (temp2 // ".tab", verify = no)
	delete ("tp*.0*", verify = no)
end
